/*
 * Simple facility to help with implementing the builtin JNI callbacks under Dolphin.
 * (Or rather, to help get around the problems caused by the mismatched threading models
 * of Dolphin and JNI.)
 *
 * Copyright Chris Uppal, 2001, 2002 and ongoing.
 */

#define DOLPHIN_JNI_HELPER_IMPLEMENTATION
#include "DolphinJNIHelper.h"

#include <stdlib.h>
#include <malloc.h>

#define WIN32_LEAN_AND_MEAN
#include <windows.h>	/* UGH!! */

#ifndef HELPER_MAX
# define HELPER_MAX		4		/* can only be changed by recompiling */
#endif

#define EXIT_CALLBACK_LOOP	0
#define VFPRINTF_CALLBACK	1
#define EXIT_CALLBACK		2
#define ABORT_CALLBACK		3

typedef struct Callback
{
	struct Callback *next;
	int		index;
	int		type;
	int		code;
	char		*buffer;
	FILE		*file;
#ifdef _DEBUG
	unsigned long	threadId;
#endif
} Callback;

static Callback *CallbackQueue,
		*MakeCallback(int, int),
		*DeQueueCallback();
static void	EnQueueCallback(Callback *),
		FreeCallback(Callback *);

static int CallbackVFPrintf(int, FILE*, const char *, va_list);
static void EnQueueCallbackVFprintf(int, FILE*, char *);
static void ReallyCallbackVFprintf(DolphinJNIHelper*, FILE *, const char *, ...);
static void CallbackExit(int, long);
static void EnQueueCallbackExit(int, long);
static void CallbackAbort(int);
static void EnQueueCallbackAbort(int);

static char *Format(const char *, va_list);

static void Debug(const char *, ...);
static const char *CallbackName(const Callback *);
#ifdef _DEBUG
# define DBPrintf(x)	Debug x
#else
# define DBPrintf(x)	/**/
#endif

static void DeQueueLoop();

#define DEFINE_HELPERS(n)						\
	static int __stdcall						\
	VFPrintf_##n(FILE *file, const char *format, va_list args)	\
	{								\
		return CallbackVFPrintf(n, file, format, args);		\
	}								\
	static void __stdcall						\
	Exit_##n(long code)						\
	{								\
		CallbackExit(n, code);					\
	}								\
	static void __stdcall						\
	Abort_##n(void)							\
	{								\
		CallbackAbort(n);					\
	}								\

#if HELPER_MAX > 0
	DEFINE_HELPERS(0)
#endif
#if HELPER_MAX > 1
	DEFINE_HELPERS(1)
#endif
#if HELPER_MAX > 2
	DEFINE_HELPERS(2)
#endif
#if HELPER_MAX > 3
	DEFINE_HELPERS(3)
#endif
#if HELPER_MAX > 4
	DEFINE_HELPERS(4)
#endif
#if HELPER_MAX > 5
	DEFINE_HELPERS(5)
#endif
#if HELPER_MAX > 6
	DEFINE_HELPERS(6)
#endif
#if HELPER_MAX > 7
	DEFINE_HELPERS(7)
#endif
#if HELPER_MAX > 8
	DEFINE_HELPERS(8)
#endif
#if HELPER_MAX > 9
	DEFINE_HELPERS(9)
#endif


#define HELPERS(n)		\
	{			\
		VFPrintf_##n,	\
		Exit_##n,	\
		Abort_##n,	\
	}	

static DolphinJNIHelper Helpers[HELPER_MAX] =
{
#if HELPER_MAX > 0
	HELPERS(0),
#endif
#if HELPER_MAX > 1
	HELPERS(1),
#endif
#if HELPER_MAX > 2
	HELPERS(2),
#endif
#if HELPER_MAX > 3
	HELPERS(3),
#endif
#if HELPER_MAX > 4
	HELPERS(4),
#endif
#if HELPER_MAX > 5
	HELPERS(5),
#endif
#if HELPER_MAX > 6
	HELPERS(6),
#endif
#if HELPER_MAX > 7
	HELPERS(7),
#endif
#if HELPER_MAX > 8
	HELPERS(8),
#endif
#if HELPER_MAX > 9
	HELPERS(9),
#endif
};


static CRITICAL_SECTION CriticalSection;
static HANDLE QueueEvent;
static HANDLE DeQueueLoopThread;



/*
 * Windows crap
 */
static DWORD WINAPI
DeQueueLoopThreadProc(LPVOID parameter)
{
	DeQueueLoop();
	CloseHandle(DeQueueLoopThread);

	return 0;
}


/*
 * more Windows crap
 */
BOOL WINAPI
DllMain(HINSTANCE handle, DWORD reason, LPVOID reserved)
{
	Callback *callback;

	switch (reason)
	{
	case DLL_PROCESS_ATTACH:
		DisableThreadLibraryCalls(handle);
		InitializeCriticalSection(&CriticalSection);
		QueueEvent = CreateEvent(NULL, FALSE, FALSE, NULL);
		break;

	case DLL_PROCESS_DETACH:
		callback = MakeCallback(EXIT_CALLBACK_LOOP, 0);
		EnQueueCallback(callback);
		CloseHandle(QueueEvent);
		break;
	}

	return TRUE;
}


/*
 * VFPrintf replacement that sends everything to the Window's system debug "stream"
 */
DLL_DECLARE long __stdcall
VFDBPrintf(FILE *file, const char *format, va_list args)
{
	char *buffer;
	int written = -1;

	if (buffer = Format(format, args))
	{
		written = strlen(buffer);
		OutputDebugString(buffer);
		free(buffer);
	}

	return written;
}


/*
 * find an unused helper slot, fill it in and return it
 */
DLL_DECLARE DolphinJNIHelper * __stdcall
GetJNIHelper(void)
{
	int i;

	EnterCriticalSection(&CriticalSection);
	if (!DeQueueLoopThread)
		DeQueueLoopThread = CreateThread(NULL, 0, DeQueueLoopThreadProc, NULL, 0, NULL);
	LeaveCriticalSection(&CriticalSection);

	for (i = 0; i < HELPER_MAX; i++)
	{
		DolphinJNIHelper *helper = NULL;

		EnterCriticalSection(&CriticalSection);
		if (!Helpers[i].thread_id)
		{
			helper = Helpers + i;
			helper->thread_id = GetCurrentThreadId();
		}
		LeaveCriticalSection(&CriticalSection);

		if (helper)
		{
			helper->vfprintf_callback = NULL;
			helper->exit_callback = NULL;
			helper->abort_callback = NULL;
			helper->queue_length = 0;
			helper->max_queue_length = 0;

			return helper;
		}
	}

	return NULL;
}


/*
 * return a helper slot to the unused pool
 */
DLL_DECLARE void __stdcall
ReleaseJNIHelper(DolphinJNIHelper * helper)
{
	int i;

	/* sanity */
	for (i = 0; i < HELPER_MAX; i++)
		if (Helpers + i == helper)
			break;
	if (i >= HELPER_MAX)
		return;

	EnterCriticalSection(&CriticalSection);
	helper->thread_id = 0;
	helper->vfprintf_callback = NULL;
	helper->exit_callback = NULL;
	helper->abort_callback = NULL;
	helper->queue_length = 0;
	helper->max_queue_length = 0;
	LeaveCriticalSection(&CriticalSection);
}


/*
 * return the total size of the helper pool
 */
DLL_DECLARE unsigned int __stdcall
MaxHelpers(void)
{
	return HELPER_MAX;
}


/*
 * return how many slots in the helper pool are currently unused
 */
DLL_DECLARE unsigned int __stdcall
AvailableHelpers(void)
{
	int i;
	unsigned int count;
	
	count = 0;
	EnterCriticalSection(&CriticalSection);
	for (i = 0; i < HELPER_MAX; i++)
		if (!Helpers[i].thread_id)
			count++;
	LeaveCriticalSection(&CriticalSection);

	return count;
}


/*
 * return the current length of the queue of unhandled callbacks for the
 * given helper
 */
DLL_DECLARE unsigned int __stdcall
CallbackQueueLength(DolphinJNIHelper *helper)
{
	unsigned int length;
	
	EnterCriticalSection(&CriticalSection);
	length = helper->queue_length;
	LeaveCriticalSection(&CriticalSection);

	return length;
}


/*
 * return the maximum length of the queue of unhandled callbacks for the
 * given helper
 */
DLL_DECLARE unsigned int __stdcall
MaxCallbackQueueLength(DolphinJNIHelper *helper)
{
	unsigned int length;
	
	EnterCriticalSection(&CriticalSection);
	length = helper->max_queue_length;
	LeaveCriticalSection(&CriticalSection);

	return length;
}


/*
 * this is called when the client issues a printf callback, we have to see
 * if we are running on the Dolphin thread, and iff so then we callback
 * into Dolphin directly, otherwise we have to extract and format the
 * data from the printf, and enqueue it so that the Dolphin thread can
 * pick it up
 */
static int
CallbackVFPrintf(int n, FILE *file, const char *format, va_list args)
{
	Callback *callback;
	DolphinJNIHelper helper;
	char *buffer;
	int size;

#ifdef _DEBUG
	buffer = Format(format, args);
	Debug("Hook (0x%X) vfprintf(\"%s\")",
		GetCurrentThreadId(),
		buffer);
	free(buffer);
#endif

	EnterCriticalSection(&CriticalSection);
	helper = Helpers[n];
	LeaveCriticalSection(&CriticalSection);

	/* sanity check, we may have been cancelled in another thread */
	if (!helper.vfprintf_callback)
		return 0;

	/* if we are on the Dolphin thread then pass on the callback directly */
	if (helper.thread_id == GetCurrentThreadId())
	{
		int retval;

		DBPrintf((
			"Hook (0x%X) calling Dolphin vfprintf directly...",
			GetCurrentThreadId()));

		retval = helper.vfprintf_callback(file, format, args);

		DBPrintf((
			"Hook (0x%X) calling Dolphin vfprintf directly -- finished",
			GetCurrentThreadId()));

		return retval;
	}

	/*
	 * copy/format vprintf data to a string buffer and enqueue the callback
	 * for the Dolphin thread to collect
	 */
	if (!(buffer = Format(format, args)))
		return -1;
	size = strlen(buffer);
	if (!(callback = MakeCallback(VFPRINTF_CALLBACK, n)))
	{
		free(buffer);
		return -1;
	}
	callback->buffer = buffer;
	callback->file = file;
	EnQueueCallback(callback);

	return size;
}


/*
 * this is called when the client issues a callback, we have to see
 * if we are running on the Dolphin thread, and iff so then we callback
 * into Dolphin directly, otherwise we have to save the exit code, and
 * enqueue it so that the Dolphin thread can pick it up
 */
static void
CallbackExit(int n, long code)
{
	Callback *callback;
	DolphinJNIHelper helper;

	DBPrintf(("Hook (0x%X) exit...", GetCurrentThreadId()));

	EnterCriticalSection(&CriticalSection);
	helper = Helpers[n];
	LeaveCriticalSection(&CriticalSection);

	/* sanity check, we may have been cancelled in another thread */
	if (!helper.exit_callback)
		return;

	/* if we are on the Dolphin thread then pass on the callback directly */
	if (helper.thread_id == GetCurrentThreadId())
	{
		DBPrintf((
			"Hook (0x%X) calling Dolphin exit directly...",
			GetCurrentThreadId()));

		helper.exit_callback(code);

		DBPrintf((
			"Hook (0x%X) calling Dolphin exit directly -- finished",
			GetCurrentThreadId()));

		return;
	}

	/* enqueue the callback for the Dolphin thread to collect */
	if (!(callback = MakeCallback(EXIT_CALLBACK, n)))
		return;
	DBPrintf(("Hook (0x%X) enqueuing exit", GetCurrentThreadId()));
	callback->code = code;
	EnQueueCallback(callback);
}


/*
 * this is called when the client issues an abort callback, we have to see
 * if we are running on the Dolphin thread, and iff so then we callback
 * into Dolphin directly, otherwise we have to enqueue it so that the Dolphin
 * thread can pick it up
 */
static void
CallbackAbort(int n)
{
	Callback *callback;
	DolphinJNIHelper helper;

	DBPrintf(("Hook (0x%X) abort...", GetCurrentThreadId()));

	EnterCriticalSection(&CriticalSection);
	helper = Helpers[n];
	LeaveCriticalSection(&CriticalSection);

	/* sanity check, we may have been cancelled in another thread */
	if (!helper.abort_callback)
		return;

	/* if we are on the Dolphin thread then pass on the callback directly */
	if (helper.thread_id == GetCurrentThreadId())
	{
		DBPrintf((
			"Hook (0x%X) calling Dolphin abort directly...",
			GetCurrentThreadId()));

		helper.abort_callback();

		DBPrintf((
			"Hook (0x%X) calling Dolphin abort directly -- finished",
			GetCurrentThreadId()));

		return;
	}

	/* enqueue the callback for the Dolphin thread to collect */
	if (!(callback = MakeCallback(ABORT_CALLBACK, n)))
		return;
	EnQueueCallback(callback);
}


/*
 * this is the main loop of the dequeue thread, it sits in a loop waiting for
 * enqueued callbacks, and when it finds one it callback into Dolphin directly.
 * NB: that notification callback does *not* happen on the main Dolphin thread,
 * it happens in this thread (that Dolphin doesn't really know about) but the
 * way that Dolphin works is that *it* sees that its in a callback from the "wrong"
 * thread and marshals the notfication over to the main thread asap where it
 * will be handled before returning here. This works provided that the main thread
 * isn't busy in -- say -- a native * C callout; if it is then it'll block until
 * that thread is ready.  This means that if the original callback happens on the
 * Dolphin thread, then we *must* pass it to Dolphin at that point, and never
 * let it reach here, or we'll deadlock
 */
static void
DeQueueLoop()
{
	int finished = 0;
	while (!finished)
	{
		Callback *callback;

		DBPrintf(("DequeueLoop (0x%X) dequeuing...", GetCurrentThreadId()));

		while (callback = DeQueueCallback())
		{
			DolphinJNIHelper helper;

			EnterCriticalSection(&CriticalSection);
			helper = Helpers[callback->index];
			LeaveCriticalSection(&CriticalSection);


			DBPrintf((
				"DequeueLoop (0x%X): dequeued: %s from thread: 0x%X",
				GetCurrentThreadId(),
				CallbackName(callback),
				callback->threadId));

			switch (callback->type)
			{
			case EXIT_CALLBACK_LOOP:
				finished = 1;
				break;

			case VFPRINTF_CALLBACK:
				// we need this extra step to convert to a proper varargs format for the callback
				ReallyCallbackVFprintf(&helper, callback->file, "%s", callback->buffer);
				free(callback->buffer);
				break;

			case EXIT_CALLBACK:
				if (helper.exit_callback)
				{
					DBPrintf(("Hook (0x%X) calling Dolphin exit indirectly...", GetCurrentThreadId()));
					helper.exit_callback(callback->code);
				}
				break;

			case ABORT_CALLBACK:
				if (helper.abort_callback)
				{
					DBPrintf(("Hook (0x%X) calling Dolphin abort indirectly...", GetCurrentThreadId()));
					helper.abort_callback();
				}
				break;
			}

			DBPrintf((
				"DequeueLoop (0x%X): dequeued: %s from thread: 0x%X -- finished",
				GetCurrentThreadId(),
				CallbackName(callback),
				callback->threadId));

			FreeCallback(callback);
		}

		DBPrintf(("DequeueLoop (0x%X) sleeping...", GetCurrentThreadId()));

		WaitForSingleObject(QueueEvent, INFINITE);
	}

	DBPrintf(("DequeueLoop (0x%X) finished", GetCurrentThreadId()));
}


static void
ReallyCallbackVFprintf(DolphinJNIHelper *helper, FILE *file, const char *format, ...)
{
	va_list	args;

	DBPrintf(("Hook (0x%X) calling Dolphin VFPrintf indirectly...", GetCurrentThreadId()));

	va_start(args, format);
	if (helper->vfprintf_callback)
		helper->vfprintf_callback(file, format, args);
	va_end(args);

	DBPrintf(("Hook (0x%X) calling Dolphin VFPrintf indirectly -- finished", GetCurrentThreadId()));
}


static Callback *
MakeCallback(int type, int index)
{
	Callback *callback = malloc(sizeof *callback);

	callback->next = NULL;
	callback->type = type;
	callback->index = index;
#ifdef _DEBUG
	callback->threadId = GetCurrentThreadId();
#endif
	return callback;
}


static void
FreeCallback(Callback *callback)
{
	free(callback);
}


static void
EnQueueCallback(Callback *callback)
{
	Callback **ptr = &CallbackQueue;
	DolphinJNIHelper *helper = Helpers + callback->index;

	DBPrintf((
		"Hook (0x%X) enqueuing %s",
		GetCurrentThreadId(),
		CallbackName(callback)));

	EnterCriticalSection(&CriticalSection);

	while (*ptr)
		ptr = &((*ptr)->next);
	*ptr = callback;

	if (++helper->queue_length > helper->max_queue_length)
		helper->max_queue_length = helper->queue_length;

	LeaveCriticalSection(&CriticalSection);

	SetEvent(QueueEvent);
}


static Callback *
DeQueueCallback()
{
	Callback *head = CallbackQueue;

	EnterCriticalSection(&CriticalSection);

	head = CallbackQueue;
	if (CallbackQueue)
	{
		CallbackQueue = CallbackQueue->next;
		--Helpers[head->index].queue_length;
	}

	LeaveCriticalSection(&CriticalSection);

	return head;
}


static char *
Format(const char *format, va_list args)
{
	char *buffer;
	int size;
	int written;

	/* copy/format vprintf data to a string buffer */
	size = 64;
	buffer = NULL;
	for (;;)
	{
		buffer = realloc(buffer, size + 1);
		if (!buffer)
			return NULL;
		written = _vsnprintf(buffer, size, format, args);
		if (written <= size && written != -1)
			break;
		size *= 2;
	}

	return buffer;
}


static void
Debug(const char *format, ...)
{
	va_list	args;

	va_start(args, format);
	VFDBPrintf(NULL, format, args);
	va_end(args);
}


static const char *
CallbackName(const Callback *callback)
{
	switch (callback->type)
	{
	case EXIT_CALLBACK_LOOP:	return "kill-demon";
	case VFPRINTF_CALLBACK:		return "vfprintf";
	case EXIT_CALLBACK:		return "exit";
	case ABORT_CALLBACK:		return "abort";
	default:			return "??ERROR??";
	}
}


static void
TestVFPrintf(int n, const char *format, ...)
{
	va_list	args;

	va_start(args, format);
	CallbackVFPrintf(n, NULL, format, args);
	va_end(args);
}


DLL_DECLARE void __stdcall
TestJNIHelperVFPrintf(DolphinJNIHelper *helper)
{
	int n = helper - Helpers;
	int i;

	for (i = 0; i < HELPER_MAX; i++)
	{
		helper = Helpers + i;
		TestVFPrintf(n,
			"Helper %d: id: 0x%X, callbacks: 0x%X 0x%X 0x%X, handlers: 0x%X 0x%X 0x%X %u[%u]\n",
			i,
			helper->thread_id,
			(unsigned long)helper->vfprintf_handler,
			(unsigned long)helper->exit_handler,
			(unsigned long)helper->abort_handler,
			(unsigned long)helper->vfprintf_callback,
			(unsigned long)helper->exit_callback,
			(unsigned long)helper->abort_callback,
			helper->queue_length,
			helper->max_queue_length);
	}
}


DLL_DECLARE void __stdcall
TestJNIHelperExit(DolphinJNIHelper *helper)
{
	int n = helper - Helpers;
	long code;

	for (code = 0; code < 10; code++)
		CallbackExit(n, code);
}


DLL_DECLARE void __stdcall
TestJNIHelperAbort(DolphinJNIHelper *helper)
{
	CallbackAbort(helper - Helpers);
}
