/*
 * Simple facility to help with implementing the builtin JNI callbacks under Dolphin.
 * (Or rather, to help get around the problems caused by the mismatched threading models
 * of Dolphin and JNI.)
 *
 * Copyright Chris Uppal, 2001, 2002 and ongoing.
 */


#ifndef DOLPHIN_JNI_HELPER_INCLUDED
#define DOLPHIN_JNI_HELPER_INCLUDED

#include <stdio.h>		/* for FILE* */
#include <stdarg.h>		/* for va_list */

#ifdef DOLPHIN_JNI_HELPER_IMPLEMENTATION
# define DLL_DECLARE	__declspec(dllexport)
#else
# define DLL_DECLARE	__declspec(dllimport)
#endif

/* use GetJNIHelper() to allocate one of these per JVM you need to talk to from Dolphin */
typedef struct DolphinJNIHelper
{
	/*
	 * pass some or all of these to JNI
	 */
	long (__stdcall * vfprintf_handler)(FILE*, const char *, va_list args);
	void (__stdcall * exit_handler)(long);
	void (__stdcall * abort_handler)(void);

	/*
	 * and then these, or such of them as are non-NULL, will be called *from a non-JNI thread*.
	 * Note that when initially allocated, these are set to NULL, which means simply ignore
	 * the relevant notification from JNI
	 */
	long (__stdcall * vfprintf_callback)(FILE*, const char *, va_list args);
	void (__stdcall * exit_callback)(long);
	void (__stdcall * abort_callback)(void);

	/*
	 * this will be set to the id of the thread that allocates the helper, if the callback
	 * from JNI happens on the same thread, then the callback is passed straight to the
	 * handler (in the same thread) rather than being queued for the background thread
	 */
	unsigned long thread_id;

	/*
	 * housekeeping
	 */
	unsigned int queue_length, max_queue_length;

} DolphinJNIHelper;


DLL_DECLARE DolphinJNIHelper *	__stdcall GetJNIHelper(void);
DLL_DECLARE void		__stdcall ReleaseJNIHelper(DolphinJNIHelper *);
DLL_DECLARE unsigned int	__stdcall MaxHelpers(void);
DLL_DECLARE unsigned int	__stdcall AvailableHelpers(void);

DLL_DECLARE unsigned int	__stdcall CallbackQueueLength(DolphinJNIHelper *);
DLL_DECLARE unsigned int	__stdcall MaxCallbackQueueLength(DolphinJNIHelper *);

DLL_DECLARE void		__stdcall TestJNIHelperVFPrintf(DolphinJNIHelper *);
DLL_DECLARE void		__stdcall TestJNIHelperExit(DolphinJNIHelper *);
DLL_DECLARE void		__stdcall TestJNIHelperAbort(DolphinJNIHelper *);


/*
 * this function has the same signature as the vfprintf hooks, and will send anything
 * the Windows' system debug stream
 */
DLL_DECLARE long		 __stdcall VFDBPrintf(FILE*, const char *, va_list args);


#endif /* DOLPHIN_JNI_HELPER_INCLUDED */
