package org.metagnostic.jniport;


/**
 * Abstract base class for requests sent to Dolphin via the DolphinRequestQueue.
 *<p>
 * The reason for all of this is to try to eliminate the problems caused by
 * the mismatch between the Dolphin and Java thread model (or, at least, to
 * reduce them to performance problems, rather than show stoppers).
 *<p>
 * Copyright &copy; 2002 and ongoing by Chris Uppal.
 *<p>
 * @author Chris Uppal (chris.uppal@metagnostic.org)
 */
public abstract class AbstractDolphinRequest
{
	/**
	 * The "tag" used to identify which Dolphin callback we want to connect to.
	 * Must not be null.
	 */
	private final Object		m_tag;

	/**
	 * The "this" of the originator of the request; optional.
	 */
	private final Object		m_originator;

	/**
	 * The parameter to the callout (probably an Object[] or similar); optional.
	 */
	private final Object		m_parameter;

	/**
	 * Latch which is set once the instance has been enqueued for evaluation.
	 */
	private boolean				m_isStarted = false;

	/**
	 * Field used to build instances into linked lists (we don't use the Java2
	 * collections to avoid doubling the number of objects allocated per
	 * callback).
	 * <p>
	 * This allows package access since it's manipulated directly by
	 * DolphinRequestQueue
	 */
	AbstractDolphinRequest		m_next;


	/**
	 * The basic constructor.
	 * @param tag	Unique object that Dolphin uses to find a request handler.
	 *				Must not be null.
	 * @param originator
	 *				Object used by convention to represent the originator of the
	 * 				request.
	 * @param parameter
	 * 				Object (may be an array or null) that is the parameter to
	 * 				the request.
	 */
	protected
	AbstractDolphinRequest(Object tag, Object originator, Object parameter)
	{
		if (tag == null)
			throw new NullPointerException("tag must not be null)");
		m_tag = tag;
		m_originator = originator;
		m_parameter = parameter;
	}


	/**
	 * Constructor.
	 * @param tag	Unique object that Dolphin uses to find a request handler.
	 * @param originator
	 *				Object used by convention to represent the originator of the
	 * 				request.
	 */
	protected
	AbstractDolphinRequest(Object tag, Object originator)
	{
		this(tag, originator, null);
	}


	/**
	 * Returns the tag provided for this request.
	 */
	public Object
	getTag()
	{
		return m_tag;
	}


	/**
	 * Returns the originator object of this request.
	 */
	public Object
	getOriginator()
	{
		return m_originator;
	}


	/**
	 * Returns the parameter provided to this request.
	 */
	public Object
	getParameter()
	{
		return m_parameter;
	}


	/**
	 * Answer whether we have started executing this callout; returns
	 * immediately.
	 */
	public synchronized boolean
	isStarted()
	{
		return m_isStarted;
	}


	/**
	 * If the evaluation of this callout is not already underway then kick it
	 * off.
	 * Returns immediately without waiting for a response; note however that
	 * it may block until Dolphin has finished processing any pending requests.
	 */
	public void
	enqueue()
	throws RequestNotHandedException
	{
		log("Request: enqueue()");
	
		DolphinNotifierThread notifier = DolphinNotifierThread.getDemon();
		if (notifier == null)
			throw new RequestNotHandedException("Dolphin is not listening");

		// note that the call to enqueue() is synched on our class object, and
		// we are calling it from a context where 'this' is also locked;
		// this iffy locking of two objects is safe only because the only code
		// which is synched on our class object are enqueue() and dequeue(),
		// neither of which call any synched methods on 'this'
		synchronized (this)
		{
			if (m_isStarted)
				return;

			m_isStarted = true;
			DolphinRequestQueue.enqueue(this);
		}
		
		// tell Dolphin that there's work waiting for it.
		// NB: this *may* block waiting for Dolphin to complete,
		// or it may just pass the notification to a background
		// thread
		notifier.notifyDolphin();
	}


	/**
	 * Return whether this request required acknowledgement.
	 */
	public boolean
	isNotificationOnly()
	{
		// default implementation is that we do not require ACKs
		return true;
	}


	/**
	 * Set the return value, and wake up any thread that is waiting for it.
	 * Only called from Dolphin (via JNI).
	 * <p>
	 * This only exists in this abstract class for convenience.
	 */
	void
	notifyCompleted(Object returnValue, Throwable thrown)
	{
		// default implementation is just to ignore it
	}


	/**
	 * Set a 'request could not be handled' error condition, and wake up
	 * any thread that is waiting for it.
	 */
	void
	notifyIgnored()
	{
		notifyCompleted(null, new RequestNotHandedException());
	}


	/**
	 * Set a 'request could not be handled' error condition, and wake up
	 * any thread that is waiting for it.
	 */
	void
	notifyIgnored(String message)
	{
		notifyCompleted(null, new RequestNotHandedException(message));
	}


	public String
	toString()
	{
		StringBuffer str = new StringBuffer(super.toString());

		str.append("(tag: ");
		str.append(m_tag);

		if (m_originator != null)
		{
			str.append(", orig: ");
			str.append(m_originator);
		}

		if (m_parameter != null)
		{
			str.append(", param: ");
			str.append(m_parameter);
		}

		if (m_isStarted)
			str.append(", started");

		str.append(")");	

		return str.toString();
	}


	protected void
	log(String message)
	{
		DolphinRequestQueue.log(message, this);
	}
}
