package org.metagnostic.jniport;


/**
 * One of the two fundamental elements of marshalling calls from Java out to a
 * controlling Dolphin Smalltalk session.
 * Instances are created to hold the data of a "callback" from Java into
 * Smalltalk.  When evaluated (send()) they are queued for Dolphin to see.
 * There is no returned value or acknowledgment.
 *<p>
 * Please note that notifications <strong>may</strong> block the sender; it
 * all depends on whether the notification is sent from the Dolphin thread
 * (in which case it does block until after the notification has been
 * recieved and processed), or from another thread (in which case it'll return
 * immediately).
 *<p>
 * @see org.metagnostic.jniport.DolphinRequest.
 *<p>
 * Copyright &copy; 2002 and ongoing by Chris Uppal.
 *<p>
 * @author Chris Uppal (chris.uppal@metagnostic.org)
 */
public class DolphinNotification
extends AbstractDolphinRequest
{
	/**
	 * The basic constructor.
	 * @param tag	Unique Object that Dolphin uses to find a request handler.
	 * @param originator
	 *				Object used by convention to represent the originator of the
	 * 				request.
	 * @param parameter
	 * 				Object (may be an array) that is the parameter to the
	 *				request.
	 */
	public
	DolphinNotification(Object tag, Object originator, Object parameter)
	{
		super(tag, originator, parameter);
	}


	/**
	 * Constructor for requests taking no parameter.
	 * @param tag	Unique Object that Dolphin uses to find a request handler.
	 * @param originator
	 *				Object used by convention to represent the originator of the
	 * 				request.
	 */
	public
	DolphinNotification(Object tag, Object originator)
	{
		super(tag, originator);
	}


	/**
	 * Convenience constructor that assumes a single boolean is passed as a
	 * Boolean.
	 * @param tag	Unique Object that Dolphin uses to find a request handler.
	 * @param originator
	 *				Object used by convention to represent the originator of the
	 * 				request.
	 * @param arg	The argument to pass.
	 */
	public
	DolphinNotification(Object tag, Object originator, boolean arg)
	{
		this(tag, originator, new Boolean(arg));	
	}


	/**
	 * Convenience constructor that assumes a single int is passed as an
	 * Integer.
	 * @param tag	Unique Object that Dolphin uses to find a request handler.
	 * @param originator
	 *				Object used by convention to represent the originator of the
	 * 				request.
	 * @param arg	The argument to pass.
	 */
	public
	DolphinNotification(Object tag, Object originator, int arg)
	{
		this(tag, originator, new Integer(arg));
	}


	/**
	 * Convenience constructor that assumes a single double is passed as a
	 * Double.
	 * @param tag	Unique Object that Dolphin uses to find a request handler.
	 * @param originator
	 *				Object used by convention to represent the originator of the
	 * 				request.
	 * @param arg	The argument to pass.
	 */
	public
	DolphinNotification(Object tag, Object originator, double arg)
	{
		this(tag, originator, new Double(arg));
	}


	/**
	 * Convenience constructor that assumes a single float is passed as a
	 * Float.
	 * @param tag	Unique Object that Dolphin uses to find a request handler.
	 * @param originator
	 *				Object used by convention to represent the originator of the
	 * 				request.
	 * @param arg	The argument to pass.
	 */
	public
	DolphinNotification(Object tag, Object originator, float arg)
	{
		this(tag, originator, new Float(arg));
	}


	/**
	 * Convenience constructor that assumes two parameters are passed in an
	 * Object array.
	 * @param tag	Unique Object that Dolphin uses to find a request handler.
	 * @param originator
	 *				Object used by convention to represent the originator of the
	 * 				request.
	 * @param arg1	The first argument to pass.
	 * @param arg2	The second argument to pass.
	 */
	public
	DolphinNotification(Object tag, Object originator, Object arg1, Object arg2)
	{
		this(tag, originator, new Object[] { arg1, arg2} );
	}


	/**
	 * Convenience constructor that assumes three parameters are passed in an
	 * Object array.
	 * @param tag	Unique Object that Dolphin uses to find a request handler.
	 * @param originator
	 *				Object used by convention to represent the originator of the
	 * 				request.
	 * @param arg1	The first argument to pass.
	 * @param arg2	The second argument to pass.
	 * @param arg3	The third argument to pass.
	 */
	public
	DolphinNotification(
			Object tag,
			Object originator,
			Object arg1,
			Object arg2,
			Object arg3)
	{
		this(tag, originator, new Object[] { arg1, arg2, arg3} );
	}


	/**
	 * Convenience constructor that assumes four parameters are passed in an
	 * Object array.
	 * @param tag	Unique Object that Dolphin uses to find a request handler.
	 * @param originator
	 *				Object used by convention to represent the originator of the
	 * 				request.
	 * @param arg1	The first argument to pass.
	 * @param arg2	The second argument to pass.
	 * @param arg3	The third argument to pass.
	 * @param arg4	The fourth argument to pass.
	 */
	public
	DolphinNotification(
			Object tag,
			Object originator,
			Object arg1,
			Object arg2,
			Object arg3,
			Object arg4)
	{
		this(tag, originator, new Object[] { arg1, arg2, arg3, arg4} );
	}


	/**
	 * Send this notification to Dolphin.  Depending on what thread this
	 * was invoked from, it <strong>may</strong> block the sender until
	 * after Dolphin has seen the notification, but there is no guarantee
	 * either way.
	 */
	public void
	send()
	throws RequestNotHandedException
	{
		enqueue();
	}
}
