package org.metagnostic.jniport;


/**
 * Helper class which defines a thread which sits in a loop trying to tell
 * Dolphin that there are pending callbacks.
 *<p>
 * Copyright &copy; 2002 and ongoing by Chris Uppal.
 *<p>
 * @author Chris Uppal (chris.uppal@metagnostic.org)
 */
class DolphinNotifierThread
extends Thread
{
	private static DolphinNotifierThread	s_demon;

	private boolean							m_serviceRequested;
	private boolean							m_shutdownRequested;
	private int								m_callbackDepth;
	private final Thread					m_dolphinNativeThread;


	/**
	 * Return the demon thread or nil if Dolphin hasn't connected yet, or
	 * has already shut us down.
	 */
	public static synchronized DolphinNotifierThread
	getDemon()
	{
		return s_demon;
	}


	/**
	 * Set the specially designated thread where we can call out to Dolphin
	 * directly; this is private since it is actually only called (via JNI)
	 * by Dolphin.
	 */
	private static synchronized void
	setDolphinNativeThread()
	{
		s_demon = new DolphinNotifierThread(currentThread());
	}


	/**
	 * Close down the request service, note that this does more than just
	 * ask the service thread to exit, it means that it can't start up
	 * again until after Dolphin next invokes setDolphinNativeThread()
	 */
	static synchronized void
	shutdown()
	{
		DolphinRequestQueue.log("Demon: shutdown()");

		if (s_demon != null)
			s_demon.requestShutdown();

		s_demon = null;
	}


	/**
	 * Create a new DolphinNotifier daemon thread, (but don't start it).
	 * Since there should normally only ever be one of these, we use a
	 * singleton pattern, hence the ctor is private.
	 */
	private
	DolphinNotifierThread(Thread thread)
	{
		super("Dolphin notifier");
		setDaemon(true);

		m_dolphinNativeThread = thread;
		m_callbackDepth = 0;
		m_serviceRequested = false;
		m_shutdownRequested = false;
	}


	/**
	 * Notify Dolphin that there may be some pending requests for it to service.
	 * Depending on whether this is called from the same thread as Dolphin runs
	 * on it will either pass the notification over to the background thread or
	 * notify Dolphin directly (in which case it will not return until the
	 * pending requests have been dealt with from this thread).
	 */
	public void
	notifyDolphin()
	{
		log("Demon: notifyDolphin()");

		// if this is the special Dolphin thread, then we *must* call directly
		// into Dolphin (or else we'll deadlock); if not then attempting to do
		// so would block this thread but not release the locks we're holding
		// (and, futhermore, would block threads that aren't interested in the
		// result of the request) so instead we'll use the background thread to
		// send the notification -- it'll get blocked, but that won't affect
		// the calling thread
		if (runningInDolphinThread())
			servicePendingRequests();
		else
			sendServiceRequest();
	}


	/**
	 * Call out to Dolphin to get it to service any outstanding requests.
	 * When this returns, all the pending items will have been dealt with.
	 * NB: This <strong>must</strong> only ever be called from the Dolphin
	 * thread
	 */
	private void
	servicePendingRequests()
	{
		log("Demon: servicePendingRequests()");

		// the logic here is tricky.  If the queue claims to have been
		// emptied at this point, then the request that caused us to enter
		// this code must already have been consumed (by Dolphin responding
		// to a notification from the demon thread).  If any other requests
		// are on the point of being enqueued then they'll end up prodding the
		// demon again, so there is no way that we can end up not sending a
		// necessary notification if we early-out here
		if (DolphinRequestQueue.queueHasBeenEmptied())
		{
			log("Demon: dolphinNotifierMethod() -- not needed");
			return;
		}

		// we need to keep count of the depth of callbacks invoked
		// via this route, since we'd prefer not to have the demon
		// be invoking callbacks at the same time as the main thead.
		// However we can't do it with normal locks since we need to
		// be able to invoke callbacks from the main thread even while
		// the background thread is already doing so!
		synchronized (this)
		{
			++m_callbackDepth;
		}

		// now call out to Dolphin.
		log("Demon: dolphinNotifierMethod() -- calling");
		dolphinNotifierMethod();
		log("Demon: dolphinNotifierMethod() -- returned");

		// let the background thread know that it's OK to use callbacks
		// again.
		// It'd be nice to postpone this notification until *after* we've
		// returned to our caller; as it is, it is possible that if there
		// any new requests pending (submitted from other threads) then
		// they'll get serviced immediately, and this thread can be starved
		// out until the other threads have stopped bombarding Dolphin with
		// their requests
		synchronized (this)
		{
			if (--m_callbackDepth < 1)
				notifyAll();
		}

		log("Demon: servicePendingRequests() -- returned");
	}


	/**
	 * Ask the demon thread to notify Dolphin that there may be some
	 * pending requests for it to service.
	 * <p>
	 * This <strong>must never</strong> be called from the Dolphin
	 * thread or from the demon thread itself.
	 * <p>
	 * Returns immediately.
	 */
	private synchronized void
	sendServiceRequest()
	{
		log("Demon: sendServiceRequest()");

		// if we haven't started running yet then do so
		if (!isAlive())
		{
			log("Demon: starting demon");
			start();
		}

		// set the flag and wake up the demon thread
		log("Demon: prodding demon");
		m_serviceRequested = true;
		notifyAll();
	}


	/**
	 * Ask the demon thread to close itself down.
	 */
	private synchronized void
	requestShutdown()
	{
		log("Demon: requestShutdown()");

		// set the flag and wake up the demon thread
		log("Demon: prodding demon");
		m_shutdownRequested = true;
		notifyAll();
	}


	/**
	 * The usual main loop of our thread.
	 */
	public void
	run()
	{
		log("Demon loop: starting");

		for (;;)
		{
			synchronized (this)
			{
				// sleep until either we've been asked to shutdown, or
				// there's a request to tell Dolphin about *and* it is
				// not already in its loop to service outstanding requests
				while (!m_shutdownRequested
				&&     !(m_serviceRequested && m_callbackDepth < 1))
				{
					try
					{
						log("Demon loop: sleeping");
						wait();
					}
					catch (InterruptedException e)
					{
					}
					log("Demon loop: woken");
				}

				// have we been closed down ?
				if (m_shutdownRequested)
				{
					log("Demon loop: shutdown requested");
					return;
				}

				// otherwise we must be able to run.
				// set the s_serviceRequested flag to false *before* we leave
				// the critical region; we won't look at it again until we next
				// go round this loop.  Note that it is only ever set to
				// false here, i.e. from this actual thread
				m_serviceRequested = false;
			}
			
			// someone has asked us to notify Dolphin

			// as in servicePendingRequests() (see the note there) we can
			// avoid sending the notification if the queue is empty at this
			// point, since whatever caused us to wake up must, by definition
			// have been handled
			if (DolphinRequestQueue.queueHasBeenEmptied())
			{
				log("Demon loop: dolphinNotifierMethod() -- not needed");
				continue;
			}

			// we now notify Dolphin, since this will block the caller (us)
			// without releasing any locks, we have to be sure that we've
			// already released them before we do so.  We don't bother
			// incr/decrementing the callback count since *we* are the
			// thread that is supposed to be blocked by them 
			log("Demon loop: dolphinNotifierMethod() -- calling");
			dolphinNotifierMethod();
			log("Demon loop: dolphinNotifierMethod() -- returned");

			// it is quite possible that some other thread has now set
			// m_serviceRequested to true; if so then we'll immediately notify
			// Dolphin again, even though it has probably already
			// cleared the requests queued from that thread -- so what ?
		}
	}


	/**
	 * Return whether we are running in the Dolphin native thread.
	 */
	public boolean
	runningInDolphinThread()
	{
		return currentThread() == m_dolphinNativeThread;
	}


	/**
	 * This is a native method, implemented by Dolphin; when it returns, all
	 * the pending items will have been dealt with.
	 * <p>
	 * Dolphin blocks the caller thread, and then executes the native code
	 * in its normal OS-level thread, and only then allows the caller to
	 * proceed.  It is important to remember that if this is called then
	 * any locks held by the caller will not be released (in the manner
	 * of wait()), and that is, at heart, the only reason for this thread
	 * to exist since it allows callers to wait() for a notification rather
	 * than leaving themselves locked, and thus potentially deadlocking
	 * Dolphin when it tries to notify them of completion.
	 */
	private static native void
	dolphinNotifierMethod();


	/**
	 * Write a message to the log stream if there is one
	 */
	private static void
	log(String message)
	{
		DolphinRequestQueue.log(message);
	}


	public String
	toString()
	{
		return getName()
			+ ", service requested:" + m_serviceRequested
			+ ", callback depth:" + m_callbackDepth;
	}
}