package org.metagnostic.jniport;


/**
 * One of the two fundamental elements of marshalling calls from Java out to a
 * controlling Dolphin Smalltalk session.
 * Instances are created to hold the data of every "callback" from Java into
 * Smalltalk.  Create one with the appropriate tag, orginator and parameter,
 * and then ask it for its value().  You can do this from any Java thread.
 * It'll block until the response has come back from the Dolphin session.
 * <p>
 * If you are running on a thread other than the main Dolphin thread, then
 * theese then can act as "promises": call startEvaluation() which will return
 * immediately, at some later time you can call value(), getReturnValue(),
 * or getException(), and you'll only then be blocked until Dolphin has
 * processed the request (if it hasn't already).
 * <p>
 * If you are running on the main thread, then you can still can
 * startEvaluation(), etc, in the same way, but you'll be blocked in
 * startEvaluation() and control won't return until Dolphin has finished
 * processing the reqest.
 *<p>
 * @see org.metagnostic.jniport.DolphinNotification.
 *<p>
 * Copyright &copy; 2002 and ongoing by Chris Uppal.
 *<p>
 * @author Chris Uppal (chris.uppal@metagnostic.org)
 */
public class
DolphinRequest
extends AbstractDolphinRequest
{
	/**
	 * Latch which is set once a return value (or exception to be re-thrown)
	 * is available.
	 */
	private boolean					m_isComplete;

	/**
	 * The returned value from the callout, may be null.  Note that attempting
	 * to retrieve this value will block the sender until a value is actually
	 * available -- that is to say that we act as "promises".
	 */
	private Object					m_returnValue;

	/**
	 * The exception (if there was one) that was thrown from the callout.
	 */
	private Throwable				m_thrown;


	/**
	 * The basic constructor.
	 * @param tag	Unique Object that Dolphin uses to find a request handler.
	 * @param originator
	 *				Object used by convention to represent the originator of the
	 * 				request.
	 * @param parameter
	 * 				Object (may be an array) that is the parameter to the
	 *				request.
	 */
	public
	DolphinRequest(Object tag, Object originator, Object parameter)
	{
		super(tag, originator, parameter);
	}


	/**
	 * Constructor for requests taking no parameter.
	 * @param tag	Unique Object that Dolphin uses to find a request handler.
	 * @param originator
	 *				Object used by convention to represent the originator of the
	 * 				request.
	 */
	public
	DolphinRequest(Object tag, Object originator)
	{
		super(tag, originator);
	}


	/**
	 * Convenience constructor that assumes a single boolean is passed as a
	 * Boolean.
	 * @param tag	Unique Object that Dolphin uses to find a request handler.
	 * @param originator
	 *				Object used by convention to represent the originator of the
	 * 				request.
	 * @param arg	The argument to pass.
	 */
	public
	DolphinRequest(Object tag, Object originator, boolean arg)
	{
		this(tag, originator, new Boolean(arg));	
	}


	/**
	 * Convenience constructor that assumes a single int is passed as an
	 * Integer.
	 * @param tag	Unique Object that Dolphin uses to find a request handler.
	 * @param originator
	 *				Object used by convention to represent the originator of the
	 * 				request.
	 * @param arg	The argument to pass.
	 */
	public
	DolphinRequest(Object tag, Object originator, int arg)
	{
		this(tag, originator, new Integer(arg));
	}


	/**
	 * Convenience constructor that assumes a single double is passed as a
	 * Double.
	 * @param tag	Unique Object that Dolphin uses to find a request handler.
	 * @param originator
	 *				Object used by convention to represent the originator of the
	 * 				request.
	 * @param arg	The argument to pass.
	 */
	public
	DolphinRequest(Object tag, Object originator, double arg)
	{
		this(tag, originator, new Double(arg));
	}


	/**
	 * Convenience constructor that assumes a single float is passed as a
	 * Float.
	 * @param tag	Unique Object that Dolphin uses to find a request handler.
	 * @param originator
	 *				Object used by convention to represent the originator of the
	 * 				request.
	 * @param arg	The argument to pass.
	 */
	public
	DolphinRequest(Object tag, Object originator, float arg)
	{
		this(tag, originator, new Float(arg));
	}


	/**
	 * Convenience constructor that assumes two parameters are passed in an
	 * Object array.
	 * @param tag	Unique Object that Dolphin uses to find a request handler.
	 * @param originator
	 *				Object used by convention to represent the originator of the
	 * 				request.
	 * @param arg1	The first argument to pass.
	 * @param arg2	The second argument to pass.
	 */
	public
	DolphinRequest(Object tag, Object originator, Object arg1, Object arg2)
	{
		this(tag, originator, new Object[] { arg1, arg2} );
	}


	/**
	 * Convenience constructor that assumes three parameters are passed in an
	 * Object array.
	 * @param tag	Unique Object that Dolphin uses to find a request handler.
	 * @param originator
	 *				Object used by convention to represent the originator of the
	 * 				request.
	 * @param arg1	The first argument to pass.
	 * @param arg2	The second argument to pass.
	 * @param arg3	The third argument to pass.
	 */
	public
	DolphinRequest(
			Object tag,
			Object originator,
			Object arg1,
			Object arg2,
			Object arg3)
	{
		this(tag, originator, new Object[] { arg1, arg2, arg3} );
	}


	/**
	 * Convenience constructor that assumes four parameters are passed in an
	 * Object array.
	 * @param tag	Unique Object that Dolphin uses to find a request handler.
	 * @param originator
	 *				Object used by convention to represent the originator of the
	 * 				request.
	 * @param arg1	The first argument to pass.
	 * @param arg2	The second argument to pass.
	 * @param arg3	The third argument to pass.
	 * @param arg4	The fourth argument to pass.
	 */
	public
	DolphinRequest(
			Object tag,
			Object originator,
			Object arg1,
			Object arg2,
			Object arg3,
			Object arg4)
	{
		this(tag, originator, new Object[] { arg1, arg2, arg3, arg4} );
	}


	/**
	 * Return whether this request required acknowledgement.
	 */
	public boolean
	isNotificationOnly()
	{
		return false;
	}


	/**
	 * Wait until Dolphin has finished processing this request (if it hasn't
	 * already done so) then return whether the callout threw.
	 */
	public boolean
	checkForException()
	{
		return getException() != null;
	}


	/**
	 * If the evaluation of this callout is not already underway then kick it
	 * off.
	 */
	public void
	startEvaluation()
	{
		try
		{
			enqueue();
		}
		catch (RequestNotHandedException e)
		{
			notifyCompleted(null, e);
		}
	}


	/**
	 * Wait until Dolphin has finished processing this request (if it hasn't
	 * already done so) then return the exception that the callout threw or
	 * null if it didn't.
	 */
	public Throwable
	getException()
	{
		startEvaluation();
		waitForResponse();

		return m_thrown;
	}


	/**
	 * Wait until Dolphin has finished processing this request (if it hasn't
	 * already done so) then return the value that the callout returned with.
	 * The result is undefined if the callour threw an exception instead of
	 * returning normally.
	 */
	public Object
	getReturnValue()
	{
		startEvaluation();
		waitForResponse();

		return m_returnValue;
	}


	/**
	 * Wait until Dolphin has finished processing this request (if it hasn't
	 * already done so) then either return the callout's return value, or
	 * throw an exception if the callout threw.
	 */
	public Object
	value()
	throws Throwable, RequestNotHandedException
	{
		startEvaluation();
		waitForResponse();

		if (m_thrown != null)
			throw m_thrown;

		return m_returnValue;
	}


	/**
	 * Answer whether Dolphin has finished executing this callout; returns
	 * immediately.
	 */
	public synchronized boolean
	isComplete()
	{
		return m_isComplete;
	}


	/**
	 * Wait until Dolphin has finished processing this request, if it hasn't
	 * already done so.
	 */
	public synchronized void
	waitForResponse()
	{
		log("Request: waitForResponse()");

		// wait until we're complete
		while (!m_isComplete)
		{
			try
			{
				log("Request: waitForResponse() -- sleeping");
				wait();
			}
			catch (InterruptedException e)
			{
			}

			log("Request: waitForResponse() -- waking");
		}

		log("Request: waitForResponse() -- returning");
	}


	/**
	 * Set the return value, and wake up any thread that is waiting for it.
	 * Only called from Dolphin (via JNI).
	 */
	synchronized void
	notifyCompleted(Object returnValue, Throwable thrown)
	{
		m_isComplete = true;
		m_returnValue = returnValue;
		m_thrown = thrown;

		log("Request: notifyCompleted()");

		notifyAll();
	}


	public String
	toString()
	{
		StringBuffer str = new StringBuffer(super.toString());

		if (m_isComplete)
		{
			if (m_thrown != null)
			{
				str.append(" threw: ");
				str.append(m_thrown);
			}
			else
			{
				str.append(" returned: ");
				str.append(m_returnValue);
			}
		}	

		return str.toString();
	}
}