package org.metagnostic.jniport;

import java.io.PrintStream;


/**
 * DolphinRequestQueue is an abstract class that acts as a "singleton" to
 * provide a queue of DolphinRequests that can be accesses from Dolphin.
 * Note that all its functionality is on the class side -- this is mainly to
 * make it easier to access from Dolphin.
 *<p>
 * Copyright &copy; 2002 and ongoing by Chris Uppal.
 *<p>
 * @author Chris Uppal (chris.uppal@metagnostic.org)
 */
public abstract class DolphinRequestQueue
{
	/**
	 * The head of a linked list of requests waiting for Dolphin to handle them.
	 */
	private static AbstractDolphinRequest	s_requestQueue;

	/**
	 * The last of a linked list of requests waiting for Dolphin to handle them.
	 */
	private static AbstractDolphinRequest	s_requestQueueTail;

	/**
	 * The current length of the request queue.
	 */
	private static int						s_requestQueueLength = 0;

	/**
	 * The maximum length of the request queue.
	 */
	private static int						s_requestQueueMax = 0;

	/**
	 * The total number of requests that have been serviced.
	 */
	private static long						s_requestsCompleted = 0;

	/**
	 * True iff Dolphin has flushed the pending queue completely, and nothing
	 * has yet been put on it.
	 */
	private static boolean					s_requestQueueFushed = false;

	/**
	 * Flag/destination to say whether we should do logging.
	 */
	private static volatile PrintStream		s_logTo = null;


	/**
	 * Returns the number of requests that are currently outstanding.
	 */
	public static synchronized int
	requestQueueLength()
	{
		return s_requestQueueLength;
	}


	/**
	 * Returns the maximum number of requests that have ever been outstanding.
	 */
	public static synchronized int
	requestQueueMax()
	{
		return s_requestQueueMax;
	}


	/**
	 * Returns the total number of requests that have been serviced.
	 */
	public static synchronized long
	requestsCompleted()
	{
		return s_requestsCompleted;
	}
	
	
	/**
	 * Returns whether Dolphin has finished processing our event queue.
	 */
	static synchronized boolean
	queueHasBeenEmptied()
	{
		return s_requestQueueFushed;
	}
	
	
	/**
	 * Write a message to the log stream if there is one
	 */
	public static void
	log(String message, AbstractDolphinRequest request)
	{
		PrintStream log = s_logTo;
		if (log == null)
			return;
		String line = ""
					+ Thread.currentThread()
					+ "\n\t" + message
					+ "\n\treq: " + request;
		log.println(line);
		log.flush();
	}


	/**
	 * Write a message to the log stream if there is one
	 */
	public static void
	log(String message)
	{
		PrintStream log = s_logTo;
		if (log == null)
			return;
		String line = ""
					+ Thread.currentThread()
					+ "\n\t" + message;
		log.println(line);
		log.flush();
	}
	
	
	/**
	 * Start logging to the named file.
	 */
	public static void
	startLogging(String filename)
	throws java.io.FileNotFoundException
	{
		startLogging(
			new java.io.PrintStream(
				new java.io.FileOutputStream(filename)));
	}


	/**
	 * Start logging to the given output stream.
	 */
	public static void
	startLogging(PrintStream out)
	{
		s_logTo = out;
	}


	/**
	 * Stop recording any log output.
	 */
	public static void
	stopLogging()
	{
		s_logTo = null;
	}


	/**
	 * Close down the request service, note that this does more than just
	 * ask the service thread to exit, it means that it can't start up
	 * again until after Dolphin next invokes setDolphinNativeThread()
	 */
	public static synchronized void
	shutdown()
	{
		DolphinRequestQueue.log("Queue: shutdown()");
		
		DolphinNotifierThread.shutdown();

		log("Queue: failing pending requests");
		AbstractDolphinRequest next;
		while ((next = dequeue()) != null)
			next.notifyIgnored("Dolphin has stopped listening");
	}


	/**
	 * Add a request to the queue that Dolphin will see when it next attempts
	 * to service any pending requests.
	 */
	static synchronized void
	enqueue(AbstractDolphinRequest request)
	{
		// NB: must not use any synched methods of any element of the queue
		// from here; see AbstractDolphinRequest.enqueue() for more discussion

		log("Queue: enqueue()", request);

		if (s_requestQueue == null)
		{
			s_requestQueue = s_requestQueueTail = request;
		}
		else
		{
			s_requestQueueTail.m_next = request;
			s_requestQueueTail = request;
		}

		s_requestQueueLength++;
		if (s_requestQueueLength > s_requestQueueMax)
			s_requestQueueMax = s_requestQueueLength;
		s_requestQueueFushed = false;
	}


	/**
	 * Remove and return the first of the current list of enqueued requests,
	 * or null if there isn't one.
	 * Only called from Dolphin (via JNI)
	 */
	private static synchronized AbstractDolphinRequest
	dequeue()
	{
		// NB: must not use any synched methods of any element of the queue
		// from here; see AbstractDolphinRequest.enqueue() for more discussion

		if (s_requestQueue == null)
		{
			log("Queue: dequeue() -- empty queue");
			s_requestQueueFushed = true;
			return null;
		}

		AbstractDolphinRequest next = s_requestQueue;
		s_requestQueue = s_requestQueue.m_next;
		next.m_next = null;
		if (next == s_requestQueueTail)
			s_requestQueueTail = null;

		--s_requestQueueLength;
		++s_requestsCompleted;

		log("Queue: dequeue()", next);

		return next;
	}
	
	
	/**
	 * Remove and return an array of up to max of the enqueued requests,
	 * or null if there aren't any.
	 * Only called from Dolphin (via JNI)
	 */
	private static synchronized AbstractDolphinRequest[]
	dequeue(int max)
	{
		// NB: must not use any synched methods of any element of the queue
		// from here; see startEvaluation() for more discussion

		if (s_requestQueue == null)
		{
			log("Queue: dequeue() -- empty queue");
			s_requestQueueFushed = true;
			return null;
		}

		AbstractDolphinRequest[] gulp
					= new AbstractDolphinRequest[
									max > s_requestQueueLength
											? s_requestQueueLength
											: max];
		for (int i = 0; i < gulp.length; ++i)
		{
			AbstractDolphinRequest next = gulp[i] = s_requestQueue;
			s_requestQueue = s_requestQueue.m_next;
			next.m_next = null;
			if (next == s_requestQueueTail)
				s_requestQueueTail = null;
		}

		s_requestQueueLength -= gulp.length;
		s_requestsCompleted += gulp.length;

		log("Queue: dequeue(" + max + ") -> " + gulp.length);

		return gulp;
	}
}