package org.metagnostic.jniport;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.EventObject;


/**
 * One of these will send a Swing (or similar) event, defined by its
 * notification method, to the Java request queue.
 * <p>
 * The fowarded Request will have its tag set to the Method, the 'originator'
 * set to the source of the Event, and param set to the Event itself.
 * <p>
 * @see org.metagnostic.jniport.DolphinRequest.
 * @see org.metagnostic.jniport.InvocationForwarder.
 * @see java.lang.reflect.InvocationHandler.
 *<p>
 * Copyright &copy; 2002 and ongoing by Chris Uppal.
 *<p>
 * @author Chris Uppal (chris.uppal@metagnostic.org)
 */
public class EventForwarder
extends InvocationForwarder
implements InvocationHandler
{
	private final Method	m_method;
	private boolean 		m_isAsynchronous;


	/**
	 * Constuct a new instance that will forward the given Method
	 * to Dolphin, and either wait for a response or not depending
	 * on whether asynch is set.
	 */
	public
	EventForwarder(Method method, boolean asynch)
	{
		m_method = method;
		m_isAsynchronous = asynch;
	}
	
	
	/**
	 * Constuct a new instance that will forward the given Method
	 * to Dolphin, and wait for a response.
	 */
	public
	EventForwarder(Method method)
	{
		this(method, true);
	}
	
	
	/**
	 * @see org.metagnostic.jniport.InvocationForwarder#invokeOther(
	 * 												Object, Method, Object[])
	 */
	protected Object
	invokeOther(Object originator, Method method, Object[] params)
	throws Throwable
	{
		if (!method.equals(m_method))
			return null;

		EventObject event = (EventObject)params[0];
		Object source = event.getSource();
		try
		{
			if (m_isAsynchronous)
			{
				new DolphinNotification(m_method, source, event).send();
				return null;
			}
			else
			{
				return new DolphinRequest(m_method, source, event).value();
			}
		}
		catch (RequestNotHandedException e)
		{
			// since we know we are messing with events, we can assume that it
			// doesn't much matter if it's not handled
			System.err.println(e);
			return null;
		}
	}


	/**
	 * Given a Method of an interface, answer an object which implements that
	 * interface and will respond to the method by forwarding it to Dolphin.
	 * Note that the method's declaring class <strong>must</strong> be an
	 * interface class, also it is assumed that the Method takes exactly
	 * one argument  which is an instance of (any subclass of)
	 * java.util.EventObject.
	 * <p>
	 * @see java.lang.reflect.Proxy
	 */
	public static Object
	forwarderFor(Method method, boolean asynch)
	{
		return Proxy.newProxyInstance(
					method.getDeclaringClass().getClassLoader(),
					new Class[] { method.getDeclaringClass() },
					new EventForwarder(method, asynch));
	}


	public String
	toString()
	{
		return this.getClass().getName() + " for " + m_method;
	}
}
