package org.metagnostic.jniport;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;


/**
 * One of these will send all "invocations" to the Java request queue
 * with the tag set to the Method, the 'originator' set to the object that
 * the method was invoked on, and param set to the Object[] array provided.
 * <p>
 * @see org.metagnostic.jniport.DolphinRequest.
 * @see java.lang.reflect.InvocationHandler.
 *<p>
 * Copyright &copy; 2002 and ongoing by Chris Uppal.
 *<p>
 * @author Chris Uppal (chris.uppal@metagnostic.org)
 */
public class InvocationForwarder
implements InvocationHandler
{
	private static Method	s_toString;
	private static Method	s_hashCode;
	private static Method	s_equals;
	static
	{
		try
		{
			s_toString = Object.class.getMethod("toString", null);
			s_hashCode = Object.class.getMethod("hashCode", null);
			s_equals = Object.class.getMethod(
										"equals",
										new Class[] { Object.class });
		}
		catch (NoSuchMethodException e)
		{
			// shouldn't be possible, and there's bugger-all we can
			// do about it if it does happen
		}
	}
	
	/**
	 * @see java.lang.reflect.InvocationHandler#invoke(Object, Method, Object[])
	 */
	public Object
	invoke(Object originator, Method method, Object[] params)
	throws Throwable
	{
		if (method.equals(s_toString))
			return invokeToString(originator);

		if (method.equals(s_hashCode))
			return new Integer(invokeHashCode(originator));

		if (method.equals(s_equals))
			return new Boolean(invokeEquals(originator, params[0]));

		return invokeOther(originator, method, params);
	}


	/**
	 * Special handler for the case where the invoked method is
	 * java.lang.Object.toString().
	 * @see java.lang.reflect.InvocationHandler#invoke(Object, Method, Object[])
	 */
	protected String
	invokeToString(Object originator)
	throws Throwable
	{
		return toString();
	}


	/**
	 * Special handler for the case where the invoked method is
	 * java.lang.Object.hashCode().
	 * @see java.lang.reflect.InvocationHandler#invoke(Object, Method, Object[])
	 */
	protected int
	invokeHashCode(Object originator)
	throws Throwable
	{
		return hashCode();
	}


	/**
	 * Special handler for the case where the invoked method is
	 * java.lang.Object.equals(Object).
	 * @see java.lang.reflect.InvocationHandler#invoke(Object, Method, Object[])
	 */
	protected boolean
	invokeEquals(Object originator, Object other)
	throws Throwable
	{
		return equals(other);
	}


	/**
	 * @see java.lang.reflect.InvocationHandler#invoke(Object, Method, Object[])
	 */
	protected Object
	invokeOther(Object originator, Method method, Object[] params)
	throws Throwable, RequestNotHandedException
	{
		return new DolphinRequest(method, originator, params).value();
	}


	/**
	 * Given an Interface, answer an object that will implements
	 * that interface and respond to its methods by forwarding them to Dolphin.
	 * Note that the method declaring class <strong>must</strong> be an
	 * interface class.
	 * @see java.lang.reflect.Proxy
	 */
	public static Object
	forwarderFor(Class clazz)
	{
		return Proxy.newProxyInstance(
					clazz.getClassLoader(),
					new Class[] { clazz },
					new InvocationForwarder());
	}


	/**
	 * Given an Array of Interfaces, answer an object that implements
	 * them and will respond to their methods by forwarding them to Dolphin.
	 * Note that the method declaring classes <strong>must</strong> be
	 * interface classes.
	 * @see java.lang.reflect.Proxy
	 */
	public static Object
	forwarderFor(Class[] classes)
	{
		return Proxy.newProxyInstance(
					classes[0].getClassLoader(),
					classes,
					new InvocationForwarder());
	}
}
