package org.metagnostic.jniport;

import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;


/**
 * One of these acts as a conduit from the Java space to the Smalltalk space.
 * Bytes written to the output stream are seen and handled (in whatever
 * way) by Dolphin.  Notice that the notifications sent to Dolphin are
 * asynchronous.
 *<p>
 * Copyright &copy; 2002 and ongoing by Chris Uppal.
 *<p>
 * @author Chris Uppal (chris.uppal@metagnostic.org)
 */
public class JavaToDolphinOutputStream
extends OutputStream
{
	// invoke the String ctor to ensure runtime uniqueness
	private static final String
			s_writeTag = new String("JavaToDolphinOutputStream.write()"),
			s_closeTag = new String("JavaToDolphinOutputStream.close()");

	// the current state of "hookedness"
	private static int							s_hookedOut, s_hookedErr;
	private static PrintStream					s_oldOut, s_oldErr;
	private static JavaToDolphinOutputStream	s_newOut, s_newErr;


	/**
	 * Replaces the current System.out stream with a PrintStream that
	 * forwards bytes to the returned instance.  If the output stream is
	 * already hooked then just incrememnts a count and returns the
	 * already installed instance
	 */
	public static synchronized JavaToDolphinOutputStream
	installAsSystemOut()
	{
		if (s_hookedOut == 0)
		{
			s_newOut = new JavaToDolphinOutputStream();
			s_oldOut = System.out;
			System.setOut(new PrintStream(s_newOut));
		}

		++s_hookedOut;

		return s_newOut;
	}


	/**
	 * Undo the effect of installAsSystemOut()
	 */
	public static synchronized void
	removeAsSystemOut()
	throws IOException
	{
		if (s_hookedOut < 1)
			return;
			
		if (--s_hookedOut == 0)
		{
			System.setOut(s_oldOut);
			s_newOut.closeNoThrow();
			s_newOut = null;
			s_oldOut = null;
		}
	}
	
	
	/**
	 * Return whether installAsSystemOut() is currently in effect.
	 */
	public static synchronized boolean
	isInstalledAsSystemOut()
	{
		return s_hookedOut > 0;
	}


	/**
	 * Replaces the current System.err stream with a PrintStream that
	 * forwards bytes to the returned instance.  If the error stream is
	 * already hooked then just incrememnts a count and returns the
	 * already installed instance
	 */
	public static JavaToDolphinOutputStream
	installAsSystemErr()
	{
		if (s_hookedErr == 0)
		{
			s_newErr = new JavaToDolphinOutputStream();
			s_oldErr = System.err;
			System.setErr(new PrintStream(s_newErr));
		}

		++s_hookedErr;

		return s_newErr;
	}


	/**
	 * Undo the effect of installAsSystemErr()
	 */
	public static synchronized void
	removeAsSystemErr()
	throws IOException
	{
		if (s_hookedErr < 1)
			return;
			
		if (--s_hookedErr == 0)
		{
			System.setErr(s_oldErr);
			s_newErr.closeNoThrow();
			s_newErr = null;
			s_oldErr = null;
		}
	}
	
	
	/**
	 * Return whether installAsSystemErr() is currently in effect.
	 */
	public static synchronized boolean
	isInstalledAsSystemErr()
	{
		return s_hookedErr > 0;
	}


	/**
	 * Returns the 'tag' used to identify write requests from this class.
	 */
	public static Object
	writeTag()
	{
		return s_writeTag;
	}


	/**
	 * Returns the 'tag' used to identify close requests from this class.
	 */
	public static Object
	closeTag()
	{
		return s_closeTag;
	}


	/**
	 * @see java.io.OutputStream#close()
	 */
	public void close()
	throws IOException
	{
		try
		{
			new DolphinNotification(s_closeTag, this).send();
		}
		catch (RequestNotHandedException e)
		{
            throw new IOException(e.getMessage());
		}
		finally
		{
			super.close();
		}
	}


	/**
	 * Same as close() but suppresses any resulting IOException.
	 * @see java.io.OutputStream#close()
	 */
	public void closeNoThrow()
	{
		try
		{
			close();
		}
		catch (IOException e)
		{
		}
	}


	/**
	 * @see java.io.OutputStream#write(int)
	 */
	public void write(int ch)
	throws IOException
	{
		write(new byte[] { (byte)ch });
	}


	/**
	 * @see java.io.OutputStream#write(byte[])
	 */
	public void write(byte[] bytes)
	throws IOException
	{
		try
		{
			new DolphinNotification(s_writeTag, this, bytes).send();
		}
		catch (RequestNotHandedException e)
		{
            throw new IOException(e.getMessage());
		}
	}


	/**
	 * @see java.io.OutputStream#write(byte[], int, int)
	 */
	public void write(byte[] bytes, int offset, int length)
	throws IOException
	{
		if (offset != 0 || length != bytes.length)
		{
			byte[] slice = new byte[length];
			System.arraycopy(bytes, offset, slice, 0, length);
			bytes = slice;
		}

		write(bytes);
	}
}
